<?php

/**
 * League.Csv (https://csv.thephpleague.com).
 *
 * (c) Ignace Nyamagana Butera <nyamsprod@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace GeminiLabs\League\Csv;

use CallbackFilterIterator;
use GeminiLabs\League\Csv\Exceptions\InvalidArgument;
use GeminiLabs\League\Csv\Exceptions\SyntaxError;
use Generator;
use Iterator;
use JsonSerializable;
use LimitIterator;

/**
 * Represents the result set of a {@link Reader} processed by a {@link Statement}.
 */
class ResultSet implements TabularDataReader, JsonSerializable
{
    /** @var Iterator The CSV records collection. */
    protected $records;
    /** @var array<string> The CSV records collection header. */
    protected $header = [];

    public function __construct(Iterator $records, array $header)
    {
        $this->validateHeader($header);

        $this->records = $records;
        $this->header = $header;
    }

    /**
     * @throws SyntaxError if the header syntax is invalid
     */
    protected function validateHeader(array $header)
    {
        if ($header !== ($filtered_header = array_filter($header, 'is_string'))) {
            throw SyntaxError::dueToInvalidHeaderColumnNames();
        }

        if ($header !== array_unique($filtered_header)) {
            throw SyntaxError::dueToDuplicateHeaderColumnNames($header);
        }
    }

    public function __destruct()
    {
        unset($this->records);
    }

    /**
     * Returns a new instance from an object implementing the TabularDataReader interface.
     * @return self
     */
    public static function createFromTabularDataReader(TabularDataReader $reader)
    {
        return new self($reader->getRecords(), $reader->getHeader());
    }

    /**
     * Returns the header associated with the result set.
     * @return array<string>
     */
    public function getHeader()
    {
        return $this->header;
    }

    /**
     * @return Iterator
     */
    public function getIterator()
    {
        return $this->getRecords();
    }

    /**
     * @return Iterator
     */
    public function getRecords(array $header = [])
    {
        $this->validateHeader($header);
        $records = $this->combineHeader($header);
        foreach ($records as $offset => $value) {
            yield $offset => $value;
        }
    }

    /**
     * Combine the header to each record if present.
     * @return Iterator
     */
    protected function combineHeader(array $header)
    {
        if ($header === $this->header || [] === $header) {
            return $this->records;
        }

        $field_count = count($header);
        $mapper = static function (array $record) use ($header, $field_count) {
            if (count($record) != $field_count) {
                $record = array_slice(array_pad($record, $field_count, null), 0, $field_count);
            }

            /** @var array<string|null> $assocRecord */
            $assocRecord = array_combine($header, $record);

            return $assocRecord;
        };

        return new MapIterator($this->records, $mapper);
    }

    /**
     * @return int
     */
    public function count()
    {
        return iterator_count($this->records);
    }

    /**
     * @return array
     */
    public function jsonSerialize()
    {
        return iterator_to_array($this->records, false);
    }

    /**
     * @param int $nth_record
     * @return array
     */
    public function fetchOne($nth_record = 0)
    {
        if ($nth_record < 0) {
            throw InvalidArgument::dueToInvalidRecordOffset($nth_record, __METHOD__);
        }

        $iterator = new LimitIterator($this->records, $nth_record, 1);
        $iterator->rewind();

        $result = $iterator->current();
        if (!is_array($result)) {
            return [];
        }

        return $result;
    }

    /**
     * @param string $name
     * @return Iterator
     * @throws UnableToProcessCsv
     */
    public function fetchColumnByName($name)
    {
        return $this->yieldColumn(
            $this->getColumnIndexByValue($name, 'name', __METHOD__)
        );
    }

    /**
     * @param int $offset
     * @return Iterator
     * @throws UnableToProcessCsv
     */
    public function fetchColumnByOffset($offset)
    {
        return $this->yieldColumn(
            $this->getColumnIndexByKey($offset, 'offset', __METHOD__)
        );
    }

    /**
     * @param int $index
     * @return Iterator
     */
    public function fetchColumn($index = 0)
    {
        return $this->yieldColumn(
            $this->getColumnIndex($index, 'offset', __METHOD__)
        );
    }

    /**
     * @param string|int $offset
     * @return Generator
     */
    protected function yieldColumn($offset)
    {
        $iterator = new MapIterator(
            new CallbackFilterIterator($this->records, function (array $record) use ($offset) {
                return isset($record[$offset]);
            }),
            function (array $record) use ($offset) {
                return $record[$offset];
            }
        );

        foreach ($iterator as $key => $value) {
            yield $key => $value;
        }
    }

    /**
     * Filter a column name against the header if any.
     *
     * @param string|int $field the field name or the field index
     * @param string $type
     * @param string $method
     *
     * @throws InvalidArgument if the field is invalid or not found
     *
     * @return string|int
     */
    protected function getColumnIndex($field, $type, $method)
    {
        if (is_string($field)) {
            return $this->getColumnIndexByValue($field, $type, $method);
        }

        return $this->getColumnIndexByKey($field, $type, $method);
    }

    /**
     * Returns the selected column name.
     *
     * @param string $value
     * @param string $type
     * @param string $method
     * @return string
     * @throws InvalidArgument if the column is not found
     */
    protected function getColumnIndexByValue($value, $type, $method)
    {
        if (false === array_search($value, $this->header, true)) {
            throw InvalidArgument::dueToInvalidColumnIndex($value, $type, $method);
        }

        return $value;
    }

    /**
     * Returns the selected column name according to its offset.
     *
     * @throws InvalidArgument if the field is invalid or not found
     *
     * @param int $index
     * @param string $type
     * @param string $method
     * @return int|string
     */
    protected function getColumnIndexByKey($index, $type, $method)
    {
        if ($index < 0) {
            throw InvalidArgument::dueToInvalidColumnIndex($index, $type, $method);
        }

        if ([] === $this->header) {
            return $index;
        }

        $value = array_search($index, array_flip($this->header), true);
        if (false === $value) {
            throw InvalidArgument::dueToInvalidColumnIndex($index, $type, $method);
        }

        return $value;
    }

    /**
     * @return Iterator
     */
    public function fetchPairs($offset_index = 0, $value_index = 1)
    {
        $offset = $this->getColumnIndex($offset_index, 'offset', __METHOD__);
        $value = $this->getColumnIndex($value_index, 'value', __METHOD__);

        $iterator = new MapIterator(
            new CallbackFilterIterator($this->records, function (array $record) use ($offset) {
                return isset($record[$offset]);
            }),
            function (array $record) use ($offset) {
                $val = $record[$value] ?? null;
                return [$record[$offset], $val];
            }
        );

        /** @var array{0:int|string, 1:string|null} $pair */
        foreach ($iterator as $pair) {
            yield $pair[0] => $pair[1];
        }
    }
}
